/**
TO DO LIST:

 - ...
 */

/*
 * Copyright (c) 2013, Cristbal Marco
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is furnished
 * to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * Event Control
 * =============
 *
 * Version: v0.43
 * Author: Cristbal Marco (makakazo)
 *
 * "Event Control" is a basic InSim application written in C/C++ using the CInsim
 * library written by Cristbal Marco. The CInsim library uses WinSock2 for the socket
 * communication, so be sure to link to it.
 *
 * This has been written highly focused on being a tutorial on programming with InSim
 * and understanding it.
 *
 * PThreads-win32 was used for this project, so be sure to link to it when compiling,
 * and place the pthreads DLL in the working directory of the executable.PThreads-win32
 * is a library that provides a POSIX pthreads API for WIN32 environments. It is licensed
 * under the LGPL. You can find info and source @ http://sourceware.org/pthreads-win32/
 */

using namespace std;

#include <iostream>
#include <fstream>
#include "pthread.h"
#include "CInsim.h"

#define IS_PRODUCT_NAME "^6Event Control v0.43"
#define MAX_PLAYERS 32

// Global objects that must be accesed by all threads
CInsim insim;

// Player struct. Contains all data related to a certain connected player.
struct player
{
    char UName[25];             // Username
    char PName[25];             // Player name
    byte UCID;                  // Connection ID
    byte PLID;                  // Player ID
    byte Admin;                 // 0 = not admin / 1 = is admin
    byte Director;              // 0 = not director / 1 = is director
    byte helpbuttons;           // Button count for the help screen
    byte cmdbuttons;            // Button count for the cmd screen
    byte flyinglap;             // Var to check if player is on a timed lap or the out lap
    byte invalidpit;            // Var to check if player pitted or spectated in an invalid way
    byte hardmsg;               // Has the hardcore session message been shown or not
};


// Main struct than contains all info stored/used by the application
struct global_info
{
    struct player players[MAX_PLAYERS];     // Array of players
    char config_data[3][256];               // [0]=IP; [1]=port; [2]=admin_password
    char directors[50][25];                 // Directors list
    char disconnections[64][25];            // People who disconnected incorrectly
    byte trackclosed;                       // Track closed or open
    byte hardcore;                          // Hardcore mode off or on
    byte off;                               // 0 = event control ON; 1 = event control OFF
};


/**
* Gathers all the required config data from config.cfg
* @param    config_data     Array to be filled with all the config data
*/
int read_config (struct global_info *ginfo)
{
    // Read config file
    char line[256];
    byte args = 0;
    ifstream cfgfile;

    cout << "\nCONFIG INFO:" << endl;
    cout << "===========" << endl;

    cfgfile.open("config.cfg", ifstream::in);

    int i=0;
    int end=0;

    // Loop while extraction from file is possible
    while ((cfgfile.good() && (!end)))
    {
        // Get line from file
        cfgfile.getline(line,256);

        if (strncmp (line,"/IP",3) == 0)
        {
            strcpy(ginfo->config_data[0], line+4);
            args++;
            cout << "HOST: " << ginfo->config_data[0] << endl;
        }
        else if (strncmp (line,"/insim_port",11) == 0)
        {
            strcpy(ginfo->config_data[1], line+12);
            args++;
            cout << "PORT: " << ginfo->config_data[1] << endl;
        }
        else if (strncmp (line,"/admin_password",15) == 0)
        {
            strcpy(ginfo->config_data[2], line+16);
            args++;
        }
        else if (strncmp (line,"/director",9) == 0)
        {
            strcpy(ginfo->directors[i], line+10);
            cout << "DIRECTOR: " << ginfo->directors[i] << endl;
            i++;
            if (i == 50)
                end = 1;
        }
    }

    cfgfile.close();

    if (args == 3)
        return 1;
    else
        return 0;
}


/**
* Reverses grid order by number provided
* @param    topx        Number of players to be reversed
*/
void eventcontrol_init(struct global_info *ginfo)
{
    // Erase all global info
    memset(ginfo, 0, sizeof(struct global_info));

    // Reload the directors
    read_config(ginfo);

    // This IS_TINY packet is used to request several things from the server before the main loop
    struct IS_TINY pack_requests;
    memset(&pack_requests, 0, sizeof(struct IS_TINY));
    pack_requests.Size = sizeof(struct IS_TINY);
    pack_requests.Type = ISP_TINY;
    pack_requests.ReqI = 1;
    pack_requests.SubT = TINY_NCN;      // Request all connections to store their user data
    insim.send_packet(&pack_requests);
    cout << "\nConnections request packet sent!" << endl;

    pack_requests.SubT = TINY_NPL;      // Request all players in-grid to know their PLID
    insim.send_packet(&pack_requests);
    cout << "Player info request packet sent!" << endl;

    cout << "\n********************************\n" << endl;
    cout << "Event Control v0.43 initialized and running!" << endl;
}


/**
* Reverses grid order by number provided
* @param    topx        Number of players to be reversed
*/
void reverse_grid(int topx)
{
    // This IS_TINY packet is used to request an IS_REO
    struct IS_TINY pack_tiny_reo;
    memset(&pack_tiny_reo, 0, sizeof(struct IS_TINY));
    pack_tiny_reo.Size = sizeof(struct IS_TINY);
    pack_tiny_reo.Type = ISP_TINY;
    pack_tiny_reo.ReqI = topx;
    pack_tiny_reo.SubT = TINY_REO;      // Request an IS_REO
    insim.send_packet(&pack_tiny_reo);
}


/**
* Clears HARDCORE SESSION button message after 5 seconds
* This function is passed as a parameter when creating the cleaning threads.
* Sleeps for 5 seconds and then clears the message.
* @param    pasUCID     UCID of the player whose messages will be cleared
*/
void *clear_msg(void *pasUCID)
{
    Sleep(5000);

    byte *UCID = (byte *)pasUCID;

    struct IS_BFN pack_bfn;
    memset(&pack_bfn, 0, sizeof(struct IS_BFN));
    pack_bfn.Size = sizeof(struct IS_BFN);
    pack_bfn.Type = ISP_BFN;
    pack_bfn.SubT = BFN_DEL_BTN;
    pack_bfn.UCID = *UCID;
    pack_bfn.ClickID = 238;         // Fixed button ID
    insim.send_packet(&pack_bfn);

    pthread_exit(NULL);

    return 0;
}


/**
* Shows an exit message to all players with a ISP_MST
* @param    ginfo       Struct that contains all the info stored by the application
* @param    notclean    0 is exit was clean and 1 if it was not
*/
void exit_message(struct global_info *ginfo, int notclean)
{
    struct IS_MST pack_mst;
    memset(&pack_mst, 0, sizeof(struct IS_MST));
    pack_mst.Size = sizeof(struct IS_MST);
    pack_mst.Type = ISP_MST;

    if (notclean)
        strncpy(pack_mst.Msg, "/msg ^1* Closing ^6Event Control ^1(error) *", 63);
    else
        strncpy(pack_mst.Msg, "/msg ^1* Closing ^6Event Control ^1(admin) *", 63);

    insim.send_packet(&pack_mst);

    Sleep(2000);
}


/**
* Sends the help screen to the specified player "splayer"
* @param    splayer    Specified player
*/
void help_screen(struct player *splayer)
{
    // 100 was chosen as length, although IS buttons can take up to 200 characters.
    char rules_text[14][100];
    strncpy(rules_text[0], "Event Control v0.43",99);

    strncpy(rules_text[1], "^3 * Welcome ",99);
    strcat(rules_text[1], splayer->PName);
    if (splayer->Admin)
        strcat(rules_text[1]," ^4(HOST ADMIN)");
    else if (splayer->Director)
        strcat(rules_text[1]," ^4(RACE DIRECTOR)");

    strncpy(rules_text[2], "^1CHAT COMMANDS:",99);
    strncpy(rules_text[3], "^3 *) Type ^7!ecommands ^3or ^7!ecmd ^3to see available commands",99);
    strncpy(rules_text[4], "^3 *) To go back to this screen type ^7!ehelp ^3or ^7!erules",99);

    strncpy(rules_text[5], "^1ABOUT EVENT CONTROL:",99);
    strncpy(rules_text[6], "^6 Event Control ^3is a tool used for directing races.",99);
    strncpy(rules_text[7], "^3 At each restart the kind of session will be announced.",99);
    strncpy(rules_text[8], "^3 During ^1HARDCORE ^3quali sessions Shift+P or spectating",99);
    strncpy(rules_text[9], "^3 is NOT allowed once you exit the pitlane for the first time.",99);
    strncpy(rules_text[10], "^3 You can telepit BEFORE exiting the pitlane though. During",99);
    strncpy(rules_text[11], "^3 ^2SOFT ^3quali sessions telepit and spectating is allowed.",99);

    strncpy(rules_text[12], "^3 Race Director messages will be shown in the chat. Read them",99);
    strncpy(rules_text[13], "^3 carefully and follow the instructions given.",99);

    // Create and fill the IS_BTN struct
    struct IS_BTN pack_btn;
    memset(&pack_btn, 0, sizeof(struct IS_BTN));
    pack_btn.Size = sizeof(struct IS_BTN);
    pack_btn.Type = ISP_BTN;
    pack_btn.ReqI = splayer->UCID;              // Must be non-zero, I'll just use UCID
    pack_btn.UCID = splayer->UCID;              // UCID of the player that will receive the button
                                                // I use buttons from 0 onward for the help screen

    // Re-fill and send the struct as many times as needed
    pack_btn.ClickID = splayer->helpbuttons++;
    pack_btn.BStyle = ISB_LIGHT;                // Light frame for main text
    pack_btn.L = 0;
    pack_btn.T = 45;
    pack_btn.W = 100;
    pack_btn.H = 108;
    insim.send_packet(&pack_btn);

    pack_btn.ClickID = splayer->helpbuttons++;
    // If no ISB_LIGHT or ISB_DARK used, then the background is transparent
    pack_btn.BStyle = ISB_DARK + ISB_LEFT + ISB_C2 + ISB_C4;   // Title text
    pack_btn.L = 0;
    pack_btn.T = 35;
    pack_btn.W = 100;
    pack_btn.H = 10;
    strcpy(pack_btn.Text, rules_text[0]);
    insim.send_packet(&pack_btn);

    int var_height = 0;
    for (int j=1; j<14; j++)
    {
        pack_btn.ClickID = splayer->helpbuttons++;
        pack_btn.BStyle = ISB_LEFT;                  // Main text lines
        pack_btn.L = 0;

        if ((j==2)||(j==5)||(j==12))
            var_height+=5;

        pack_btn.T = 45 + j*5 + var_height;
        pack_btn.W = 100;
        pack_btn.H = 5;
        strcpy(pack_btn.Text, rules_text[j]);
        insim.send_packet(&pack_btn);
    }

    pack_btn.ClickID = 99;                                      // "Close" button
    pack_btn.BStyle = ISB_DARK + ISB_CLICK + ISB_C1 + ISB_C4;   // dark + clickable + red
    pack_btn.L = 80;
    pack_btn.T = 145;
    pack_btn.W = 15;
    pack_btn.H = 6;
    strcpy(pack_btn.Text, "close");
    insim.send_packet(&pack_btn);
}


/**
* Clears all the buttons from the help screen for the given player
* @param    splayer    Specified player
*/
void clear_help_screen(struct player *splayer)
{
    // Create and fill an IS_BFN packet
    struct IS_BFN pack_bfn;
    memset(&pack_bfn, 0, sizeof(struct IS_BFN));
    pack_bfn.Size = sizeof(struct IS_BFN);
    pack_bfn.Type = ISP_BFN;
    pack_bfn.SubT = BFN_DEL_BTN;
    pack_bfn.UCID = splayer->UCID;

    // Clear all buttons from the cmd screen. This won't clear split or difference messages.
    //cout << "(Clearing buttons from help screen...)" << endl;
    pack_bfn.ClickID = 99;
    insim.send_packet(&pack_bfn);

    for(int j=splayer->helpbuttons; j>0; j--)
    {
        pack_bfn.ClickID = j-1;
        insim.send_packet(&pack_bfn);
    }
    splayer->helpbuttons = 0;
}


/**
* Shows the chat commands to the specified player "splayer"
* @param    splayer    Specified player
*/
void cmd_screen(struct player *splayer)
{
    // 64 is the maximum text length allowed by IS_MTC packets
    char commands_text[20][100];
    strncpy(commands_text[0], "CHAT COMMANDS:",63);
    strncpy(commands_text[1], "^1BASIC COMMANDS:",63);
    strncpy(commands_text[2], "^3 *) ^7!ehelp ^3or ^7!erules : ^3Shows help and rules",63);         // Help and rules
    strncpy(commands_text[3], "^3 *) ^7!ecommands ^3or ^7!ecmd : ^3This commands list",63);         // Commands list
    strncpy(commands_text[4], "^1DIRECTOR COMMANDS:",63);
    strncpy(commands_text[5], "^3 *) ^7!em [message] : ^3Preformatted Race Director message",63);   // Race Director message
    strncpy(commands_text[6], "^3 *) ^7!eopen ^3or ^7!eop : ^2Open ^3track, everybody allowed",63); // Open track
    strncpy(commands_text[7], "^3 *) ^7!eclose ^3or ^7!ecl : ^1Close ^3track, everybody specs",63); // Close track
    strncpy(commands_text[8], "^3 *) ^7!esoftq ^3or ^7!esq : ^3Start soft qualify",63);             // Soft qual
    strncpy(commands_text[9], "^3 *) ^7!ehardq ^3or ^7!ehq : ^3Start hardcore qualify",63);         // Hard qual
    strncpy(commands_text[10], "^3 *) ^7!erace ^3or ^7!erc : ^3Restart race",63);                    // Restart race
    strncpy(commands_text[11], "^3 *) ^7!eend : ^3End race, to lobby screen",63);                   // End race
    strncpy(commands_text[12], "^3 *) ^7!erev [opt] : ^3Lobby, reverse [opt] positions (Def 8)",63);// Reverse top grid
    strncpy(commands_text[13], "^3 *) ^7!espec, !ecars, !etrack, !eweather, !ewind, !equal, !elaps, !ehours",99);    // Several LFS commands
    strncpy(commands_text[14], "^3 *) ^7!emustpit, !ecanreset, !efcv, !emidrace, !eclear, !ep_dt, !ep_sg, !ep_30",99); // Several LFS commands
    strncpy(commands_text[15], "^3 *) ^7!ep_45, !ep_clear, !ercm, !ercm_ply, !ercm_all, !ercc_ply, !ercc_all",99); // Several LFS commands
    strncpy(commands_text[16], "^1ADMIN COMMANDS:",63);
    strncpy(commands_text[17], "^3 *) ^7!eon : ^3Switch ^6Event Control ^3ON",63);                  // Switch ON
    strncpy(commands_text[18], "^3 *) ^7!eoff : ^3Switch ^6Event Control ^3OFF",63);                // Switch OFF
    strncpy(commands_text[19], "^3 *) ^7!eexit ^3or ^7!eex : ^3Exit ^6Event Control",63);           // Exit

    // Create and fill the IS_BTN struct
    struct IS_BTN pack_btn;
    memset(&pack_btn, 0, sizeof(struct IS_BTN));
    pack_btn.Size = sizeof(struct IS_BTN);
    pack_btn.Type = ISP_BTN;
    pack_btn.ReqI = splayer->UCID;                      // Must be non-zero, I'll just use UCID
    pack_btn.UCID = splayer->UCID;                      // UCID of the player that will receive the button

    pack_btn.ClickID = 100 + (splayer->cmdbuttons++);
    pack_btn.BStyle = ISB_DARK + ISB_LEFT + ISB_C2 + ISB_C4;    // Title text
    pack_btn.L = 0;
    pack_btn.T = 35;
    pack_btn.W = 100;
    pack_btn.H = 10;
    strcpy(pack_btn.Text, commands_text[0]);
    insim.send_packet(&pack_btn);

    int j;
    for (j=1; j<4; j++)
    {
        pack_btn.ClickID = 100 + (splayer->cmdbuttons++);
        pack_btn.BStyle = ISB_LEFT;             // Main text lines
        pack_btn.L = 0;
        pack_btn.T = 40 + j*5;
        pack_btn.W = 100;
        pack_btn.H = 5;
        strcpy(pack_btn.Text, commands_text[j]);
        insim.send_packet(&pack_btn);
    }
    // Messages only for Directors
    if ((splayer->Admin) || (splayer->Director))
    {
        for (j=4; j<16; j++)
        {
            pack_btn.ClickID = 100 + (splayer->cmdbuttons++);
            pack_btn.BStyle = ISB_LEFT;             // Main text lines
            pack_btn.L = 0;
            pack_btn.T = 45 + j*5;
            pack_btn.W = 100;
            pack_btn.H = 5;
            strcpy(pack_btn.Text, commands_text[j]);
            insim.send_packet(&pack_btn);
        }
    }
    // Messages only for Admins
    if (splayer->Admin)
    {
        for (j=16; j<20; j++)
        {
            pack_btn.ClickID = 100 + (splayer->cmdbuttons++);
            pack_btn.BStyle = ISB_LEFT;             // Main text lines
            pack_btn.L = 0;
            pack_btn.T = 50 + j*5;
            pack_btn.W = 100;
            pack_btn.H = 5;
            strcpy(pack_btn.Text, commands_text[j]);
            insim.send_packet(&pack_btn);
        }
    }

    // Re-fill and send the struct as many times as needed
    pack_btn.ClickID = 100 + (splayer->cmdbuttons++);           // I use buttons from 100 onward for the cmd screen
    pack_btn.BStyle = ISB_LIGHT;                                // Light frame for main text
    pack_btn.L = 0;
    pack_btn.T = 45;
    pack_btn.W = 100;
    pack_btn.H = 13 + j*5;
    memset(pack_btn.Text, 0, sizeof(pack_btn.Text));
    insim.send_packet(&pack_btn);

    pack_btn.ClickID = 199;          // "Close" button
    pack_btn.BStyle = ISB_DARK + ISB_CLICK + ISB_C1 + ISB_C4;   // dark + clickable + red
    pack_btn.L = 80;
    pack_btn.T = 50 + j*5;
    pack_btn.W = 15;
    pack_btn.H = 6;
    strcpy(pack_btn.Text, "close");
    insim.send_packet(&pack_btn);
}


/**
* Clears all the buttons from the cms screen for the given player
* @param    splayer    Specified player
*/
void clear_cmd_screen(struct player *splayer)
{
    // Create and fill an IS_BFN packet
    struct IS_BFN pack_bfn;
    memset(&pack_bfn, 0, sizeof(struct IS_BFN));
    pack_bfn.Size = sizeof(struct IS_BFN);
    pack_bfn.Type = ISP_BFN;
    pack_bfn.SubT = BFN_DEL_BTN;
    pack_bfn.UCID = splayer->UCID;

    // Clear all buttons from the cmd screen. This won't clear split or difference messages.
    //cout << "(Clearing buttons from cmd screen...)" << endl;
    pack_bfn.ClickID = 199;
    insim.send_packet(&pack_bfn);

    for(int j=splayer->cmdbuttons; j>0; j--)
    {
        pack_bfn.ClickID = 100 + (j-1);
        insim.send_packet(&pack_bfn);
    }
    splayer->cmdbuttons = 0;
}


/**
* Erases the flying lap button message for the player specified
* @param    UCID   player that will receive the clear message
*/
void erase_flyinglap(struct player *splayer)
{
    struct IS_BFN pack_bfn;
    memset(&pack_bfn, 0, sizeof(struct IS_BFN));
    pack_bfn.Size = sizeof(struct IS_BFN);
    pack_bfn.Type = ISP_BFN;
    pack_bfn.SubT = BFN_DEL_BTN;
    pack_bfn.UCID = splayer->UCID;
    pack_bfn.ClickID = 237;
    insim.send_packet(&pack_bfn);
}


/**
* Erases the no pit allowed button message for the player specified
* @param    UCID   player that will receive the clear message
*/
void erase_nopit(struct player *splayer)
{
    struct IS_BFN pack_bfn;
    memset(&pack_bfn, 0, sizeof(struct IS_BFN));
    pack_bfn.Size = sizeof(struct IS_BFN);
    pack_bfn.Type = ISP_BFN;
    pack_bfn.SubT = BFN_DEL_BTN;
    pack_bfn.UCID = splayer->UCID;
    pack_bfn.ClickID = 236;
    insim.send_packet(&pack_bfn);
}


/**
* Actions performed when a button is clicked.
* There are two "close" clickable button. One in the help screen and another in
* +the invalid rejoin screen. We have to check the ClickID.
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_btc (struct global_info *ginfo)
{
    //cout << " * IS_BTC (button click)" << endl;
    int i;

    struct IS_BTC *pack_btc = (struct IS_BTC*)insim.get_packet();

    // Find the player
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_btc->UCID)
        {
            //cout << "ginfo->players[" << i << "].UCID found" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }

    // Create and fill an IS_BFN packet
    struct IS_BFN pack_bfn;
    memset(&pack_bfn, 0, sizeof(struct IS_BFN));
    pack_bfn.Size = sizeof(struct IS_BFN);
    pack_bfn.Type = ISP_BFN;
    pack_bfn.SubT = BFN_DEL_BTN;
    pack_bfn.UCID = ginfo->players[i].UCID;

    if (pack_btc->ClickID == 232)
    {
        // Clear all buttons from the invalid rejoin screen
        //cout << "(Clearing buttons from invalid rejoin screen...)" << endl;
        pack_bfn.ClickID = 235;
        insim.send_packet(&pack_bfn);
        pack_bfn.ClickID = 234;
        insim.send_packet(&pack_bfn);
        pack_bfn.ClickID = 233;
        insim.send_packet(&pack_bfn);
        pack_bfn.ClickID = 232;
        insim.send_packet(&pack_bfn);
    }
    else if (pack_btc->ClickID == 99) // Button ID corresponds to close help screen
    {
        // Clear all buttons from the help screen. This won't clear split or difference messages.
        clear_help_screen(&ginfo->players[i]);
    }
    else if (pack_btc->ClickID == 199) // Button ID corresponds to close cmd screen
    {
        // Clear all buttons from the cmd screen. This won't clear split or difference messages.
        clear_cmd_screen(&ginfo->players[i]);
    }
}


/**
* Connection left
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_cnl (struct global_info *ginfo)
{
    //cout << " * IS_CNL (connection left)" << endl;
    int i;

    struct IS_CNL *pack_cnl = (struct IS_CNL*)insim.get_packet();

    // Find player and set the whole player struct he was using to 0
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_cnl->UCID)
        {
            //cout << "ginfo->players[" << i << "].UCID found: exiting" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;

            if (ginfo->players[i].invalidpit != 0)
            {
                for (int j=0; j<64; j++)
                {
                    if (strncmp(ginfo->disconnections[j], "", 1) == 0)
                    {
                        strcpy(ginfo->disconnections[j], ginfo->players[i].UName);
                        break;
                    }
                }
            }

            memset(&ginfo->players[i], 0, sizeof(ginfo->players[i]));
            //cout << "ginfo->players[" << i << "].UCID found: CHECKING" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }
}


/**
* Player changed his pilot name (player name). Update the affected data.
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_cpr (struct global_info *ginfo)
{
    //cout << " * IS_CPR (player rename)" << endl;
    int i;

    struct IS_CPR *pack_cpr = (struct IS_CPR*)insim.get_packet();

    // Find the player
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_cpr->UCID)
        {
            //cout << "ginfo->players[" << i << "].PLID found" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }

    // Update the player name
    strcpy(ginfo->players[i].PName, pack_cpr->PName);
    //cout << "(Player changed name to " << ginfo->players[i].PName << ")" << endl;

}


/**
* Actions performed when a player completes a lap
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_lap (struct global_info *ginfo)
{
    //cout << " * IS_LAP (lap completed)" << endl;
    int i;

    struct IS_LAP *pack_lap = (struct IS_LAP*)insim.get_packet();

    // Find the player
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].PLID == pack_lap->PLID)
        {
            //cout << "ginfo->players[" << i << "].PLID found" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }

    // If player was on his out lap, clear the warning message
    if (ginfo->players[i].flyinglap == 1)
    {
        ginfo->players[i].flyinglap = 0;
        erase_flyinglap(&ginfo->players[i]);
    }

}


/**
* Actions performed when someone types something in the chat (packet IS_MSO)
* @param    ginfo   Struct that contains all the info stored by the application
* @return   1 in normal cases, -1 if the super admin requested !eexit
*/
int case_mso (struct global_info *ginfo)
{
    //cout << " * IS_MSO (chat message)" << endl;
    int i;

    struct IS_MSO *pack_mso = (struct IS_MSO*)insim.get_packet();

    // The chat message is sent by the host, don't do anything
    if (pack_mso->UCID == 0)
    {
        //cout << "(Chat message by host: " << pack_mso->Msg + ((unsigned char)pack_mso->TextStart) << ")" << endl;
        return 1;
    }

    // Find the player that wrote in the chat
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_mso->UCID)
        {
            //cout << "ginfo->players[" << i << "].UCID found" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            //cout << "Msg: " << pack_mso->Msg << endl;
            break;
        }
    }

    if (!(ginfo->off))
    {
        // !ehelp
        if ((strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ehelp") == 0) || (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!erules") == 0))
        {
            //cout << "(!ehelp requested)" << endl;
            if (ginfo->players[i].cmdbuttons != 0)
                clear_cmd_screen(&ginfo->players[i]);

            help_screen(&ginfo->players[i]);
            return 1;
        }
        // !ecommands
        else if ((strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ecommands") == 0) || (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ecmd") == 0))
        {
            //cout << "(!ecommands requested)" << endl;
            if (ginfo->players[i].helpbuttons != 0)
                clear_help_screen(&ginfo->players[i]);

            cmd_screen(&ginfo->players[i]);
            return 1;
        }

        // These commands can be executed only by directors
        if ((ginfo->players[i].Director) || (ginfo->players[i].Admin))
        {
            // Race messages by directors
            if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!em ", 4) == 0)
            {
                // Send /msg message to announce that track is open
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;

                for (int j = 0; pack_mso->Msg[j] != '\0'; j++){
                    pack_mso->Msg[j] = toupper(pack_mso->Msg[j]);
                }

                strncpy(pack_mst.Msg, "/msg ^6 * DIRECTOR: ^1", 63);
                strncpy(pack_mst.Msg+22, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 4, 41);
                insim.send_packet(&pack_mst);
            }
            // Open the track. Everybody allowed in
            else if ((strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eopen") == 0) || (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eop") == 0))
            {
                // Reset everybody's invalidpit status to zero
                //cout << "Reseting everybody's invalidpit status to 0 (zero)" << endl;
                for (int j=0; j < MAX_PLAYERS; j++)
                    ginfo->players[j].invalidpit = 0;

                // Set trackclosed to 0 and hardcore to 0
                ginfo->trackclosed = 0;
                ginfo->hardcore = 0;
                //cout << "ginfo->trackclosed: 0" << endl;
                //cout << "ginfo->hardcore: 0" << endl;

                // Clear possible buttons for everybody
                for (int i=0; i < MAX_PLAYERS; i++)
                {
                    if (ginfo->players[i].UCID != 0)
                    {
                        erase_flyinglap(&ginfo->players[i]);
                        erase_nopit(&ginfo->players[i]);
                    }
                }


                // Send /msg message to announce that track is open
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** THE TRACK IS NOW ^2OPEN ^3***", 63);
                insim.send_packet(&pack_mst);
            }
            // Close the track. Everybody sent to spectators and nobody allowed in
            else if ((strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eclose") == 0) || (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ecl") == 0))
            {
                // Reset everybody's invalidpit status to one
                //cout << "Reseting everybody's invalidpit status to 1 (one)" << endl;
                for (int j=0; j < MAX_PLAYERS; j++)
                    ginfo->players[j].invalidpit = 1;

                // Set trackclosed to 1
                ginfo->trackclosed = 1;
                //cout << "ginfo->trackclosed: 1" << endl;

                // Send /msg message to announce that track is open
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** THE TRACK IS NOW ^1CLOSED ^3***", 63);
                insim.send_packet(&pack_mst);

                // Everybody to spectators
                for (int i=0; i < MAX_PLAYERS; i++)
                {
                    if (ginfo->players[i].UCID != 0)
                    {
                        strcpy(pack_mst.Msg, "/spec ");
                        strcpy(pack_mst.Msg + 6, ginfo->players[i].PName);
                        //cout << pack_mst.Msg << endl;
                        insim.send_packet(&pack_mst);
                    }
                }
            }
            // Reset qualify as soft
            else if ((strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!esoftq") == 0) || (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!esq") == 0))
            {
                // Reset everybody's invalidpit status to zero
                //cout << "Reseting everybody's invalidpit status to 0 (zero)" << endl;
                for (int i=0; i < MAX_PLAYERS; i++)
                {
                    ginfo->players[i].invalidpit = 0;
                }

                // Set trackclosed to 0 and hardcore to 0
                ginfo->trackclosed = 0;
                ginfo->hardcore = 0;
                //cout << "ginfo->trackclosed: 0" << endl;
                //cout << "ginfo->hardcore: 0" << endl;

                // Send /msg message to announce that qualifying is restarting
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** RESTARTING SOFT QUALIFY! ***", 63);
                insim.send_packet(&pack_mst);
                // Restart qualify
                strncpy(pack_mst.Msg, "/qualify", 63);
                insim.send_packet(&pack_mst);
            }
            // Reset qualify as hardcore
            else if ((strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ehardq") == 0) || (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ehq") == 0))
            {
                // Reset everybody's invalidpit status to zero
                //cout << "Reseting everybody's invalidpit status to 0 (zero)" << endl;
                for (int i=0; i < MAX_PLAYERS; i++)
                {
                    ginfo->players[i].invalidpit = 0;
                }

                // Set trackclosed to 0 and hardcore to 1
                ginfo->trackclosed = 0;
                ginfo->hardcore = 1;
                //cout << "ginfo->trackclosed: 0" << endl;
                //cout << "ginfo->hardcore: 1" << endl;

                // Send /msg message to announce that qualifying is restarting
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** RESTARTING HARD QUALIFY! ***", 63);
                insim.send_packet(&pack_mst);
                // Restart qualify
                strncpy(pack_mst.Msg, "/qualify", 63);
                insim.send_packet(&pack_mst);
            }
            // Restart race
            else if ((strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!erace") == 0) || (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!erc") == 0))
            {
                // Send /msg message to announce that race is restarting
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** RESTARTING RACE!!! ***", 63);
                insim.send_packet(&pack_mst);
                // Restart race
                strncpy(pack_mst.Msg, "/restart", 63);
                insim.send_packet(&pack_mst);
            }
            // End race
            else if (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eend") == 0)
            {
                // Send /msg message to announce that race is ending
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** TO LOBBY SCREEN! ***", 63);
                insim.send_packet(&pack_mst);
                // Restart race
                strncpy(pack_mst.Msg, "/end", 63);
                insim.send_packet(&pack_mst);
            }
            // Reverse grid by number specified
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!erev", 5) == 0)
            {
                // Default number of reversed drivers
                int revx=8;
                char rev_char[64];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 6)
                {
                    strcpy(rev_char, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 6);
                    revx = atoi(rev_char);

                    if ((revx<2) || (revx > 32))
                        revx = 0;

                    //cout << "\n NUMERO SOLICITADO: " << revx << endl;
                }

                // Call the reordering function
                reverse_grid(revx);
            }
            // Send player to spectators
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!espec ", 7) == 0)
            {
                char license[25];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(license, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    // Send player to spectators
                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/spec ", 63);
                    strcat(pack_mst.Msg, license);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set cars
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ecars ", 7) == 0)
            {
                char cars[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(cars, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/cars ", 63);
                    strcat(pack_mst.Msg, cars);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set track
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!etrack ", 8) == 0)
            {
                char track[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 8)
                {
                    strcpy(track, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 8);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/track ", 63);
                    strcat(pack_mst.Msg, track);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set weather
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eweather ", 10) == 0)
            {
                char weather[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 10)
                {
                    strcpy(weather, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 10);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/weather ", 63);
                    strcat(pack_mst.Msg, weather);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set wind
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ewind ", 7) == 0)
            {
                char wind[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(wind, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/wind ", 63);
                    strcat(pack_mst.Msg, wind);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set qual minutes
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!equal ", 7) == 0)
            {
                char qual[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(qual, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/qual ", 63);
                    strcat(pack_mst.Msg, qual);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set race laps
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!elaps ", 7) == 0)
            {
                char laps[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(laps, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/laps ", 63);
                    strcat(pack_mst.Msg, laps);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set race hours
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ehours ", 8) == 0)
            {
                char hours[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 8)
                {
                    strcpy(hours, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 8);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/hours ", 63);
                    strcat(pack_mst.Msg, hours);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set mustpit value
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!emustpit ", 10) == 0)
            {
                char mustpit[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 10)
                {
                    strcpy(mustpit, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 10);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/mustpit ", 63);
                    strcat(pack_mst.Msg, mustpit);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set canreset value
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ecanreset ", 11) == 0)
            {
                char canreset[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 11)
                {
                    strcpy(canreset, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 11);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/canreset ", 63);
                    strcat(pack_mst.Msg, canreset);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set midrace join
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!emidrace ", 10) == 0)
            {
                char midrace[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 10)
                {
                    strcpy(midrace, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 10);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/midrace ", 63);
                    strcat(pack_mst.Msg, midrace);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set fcv
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!efcv ", 6) == 0)
            {
                char fcv[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 6)
                {
                    strcpy(fcv, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 6);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/fcv ", 63);
                    strcat(pack_mst.Msg, fcv);
                    insim.send_packet(&pack_mst);
                }
            }
            // Clear the grid (only in lobby screen)
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eclear", 7) == 0)
            {
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** GRID CLEARED! ***", 63);
                insim.send_packet(&pack_mst);
                strncpy(pack_mst.Msg, "/clear", 63);
                insim.send_packet(&pack_mst);
            }
            // Set p_dt (drive through penalty)
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ep_dt ", 7) == 0)
            {
                char p_dt[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(p_dt, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/p_dt ", 63);
                    strcat(pack_mst.Msg, p_dt);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set p_sg (stop and go penalty)
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ep_sg ", 7) == 0)
            {
                char p_sg[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(p_sg, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/p_sg ", 63);
                    strcat(pack_mst.Msg, p_sg);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set p_30 (30 seconds penalty)
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ep_30 ", 7) == 0)
            {
                char p_30[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(p_30, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/p_30 ", 63);
                    strcat(pack_mst.Msg, p_30);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set p_45 (45 seconds penalty)
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ep_45 ", 7) == 0)
            {
                char p_45[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 7)
                {
                    strcpy(p_45, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 7);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/p_45 ", 63);
                    strcat(pack_mst.Msg, p_45);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set p_clear (clear penalty)
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ep_clear ", 10) == 0)
            {
                char p_clear[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 10)
                {
                    strcpy(p_clear, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 10);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/p_clear ", 63);
                    strcat(pack_mst.Msg, p_clear);
                    insim.send_packet(&pack_mst);
                }
            }
            // Set rcm text
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ercm ", 6) == 0)
            {
                char rcm[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 6)
                {
                    strcpy(rcm, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 6);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/rcm ", 63);
                    strcat(pack_mst.Msg, rcm);
                    insim.send_packet(&pack_mst);
                }
            }
            // Send rcm to username
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ercm_ply ", 10) == 0)
            {
                char rcm_ply[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 10)
                {
                    strcpy(rcm_ply, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 10);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/rcm_ply ", 63);
                    strcat(pack_mst.Msg, rcm_ply);
                    insim.send_packet(&pack_mst);
                }
            }
            // Send rcm to all players
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ercm_all", 9) == 0)
            {
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/rcm_all", 63);
                insim.send_packet(&pack_mst);
            }
            // Clear rcm for username
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ercc_ply ", 10) == 0)
            {
                char rcc_ply[63];

                if (strlen(pack_mso->Msg + ((unsigned char)pack_mso->TextStart)) > 10)
                {
                    strcpy(rcc_ply, pack_mso->Msg + ((unsigned char)pack_mso->TextStart) + 10);

                    struct IS_MST pack_mst;
                    memset(&pack_mst, 0, sizeof(struct IS_MST));
                    pack_mst.Size = sizeof(struct IS_MST);
                    pack_mst.Type = ISP_MST;
                    strncpy(pack_mst.Msg, "/rcc_ply ", 63);
                    strcat(pack_mst.Msg, rcc_ply);
                    insim.send_packet(&pack_mst);
                }
            }
            // Clear rcm for all players
            else if (strncmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!ercc_all", 9) == 0)
            {
                struct IS_MST pack_mst;
                memset(&pack_mst, 0, sizeof(struct IS_MST));
                pack_mst.Size = sizeof(struct IS_MST);
                pack_mst.Type = ISP_MST;
                strncpy(pack_mst.Msg, "/rcc_all", 63);
                insim.send_packet(&pack_mst);
            }
        }
    }

    // These commands can be executed only by admins
    if (ginfo->players[i].Admin)
    {
        // !eexit can only be requested by "admins"
        if ((strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eexit") == 0) || (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eex") == 0))
        {
            //cout << "(!eexit requested)" << endl;
            exit_message(ginfo,0);

            return -1;
        }
        // !eon turns Event Control ON
        else if (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eon") == 0)
        {
            ginfo->off = 0;

            cout << "\n * Event Control turned ON! * " << endl;

            // Initialize Event Control
            eventcontrol_init(ginfo);

            // Send /msg message to announce that Event Control has been turned ON
            struct IS_MST pack_mst;
            memset(&pack_mst, 0, sizeof(struct IS_MST));
            pack_mst.Size = sizeof(struct IS_MST);
            pack_mst.Type = ISP_MST;
            strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** EVENT CONTROL ^2ON ^3***", 63);
            insim.send_packet(&pack_mst);
        }
        // !eoff turns Event Control OFF
        else if (strcmp(pack_mso->Msg + ((unsigned char)pack_mso->TextStart), "!eoff") == 0)
        {
            ginfo->off = 1;

            cout << "\n * Event Control turned OFF! * " << endl;

            // Erase all buttons by this insim instance
            struct IS_BFN pack_bfn;
            memset(&pack_bfn, 0, sizeof(struct IS_BFN));
            pack_bfn.Size = sizeof(struct IS_BFN);
            pack_bfn.Type = ISP_BFN;
            pack_bfn.SubT = BFN_CLEAR;
            pack_bfn.UCID = 255;
            insim.send_packet(&pack_bfn);

            // Send /msg message to announce that Event Control has been turned OFF
            struct IS_MST pack_mst;
            memset(&pack_mst, 0, sizeof(struct IS_MST));
            pack_mst.Size = sizeof(struct IS_MST);
            pack_mst.Type = ISP_MST;
            strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** EVENT CONTROL ^1OFF ^3***", 63);
            insim.send_packet(&pack_mst);
        }

    }
    return 1;
}


/**
* New connection. Add the player to the list and send welcome text and help screen.
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_ncn (struct global_info *ginfo)
{
    //cout << " * IS_NCN (new connection)" << endl;
    int i;

    struct IS_NCN *pack_ncn = (struct IS_NCN*)insim.get_packet();

    // UCID == 0 means it's the host, don't add it to the array.
    // I use UCID=0 for empty slots within the player array.
    if (pack_ncn->UCID == 0)
    {
        //cout << "(Host connected, not adding him to array...)" << endl;
        return;
    }

    // Stop when the first empty slot is found, checking UCID==0
    for (i=0; i<MAX_PLAYERS; i++)
        if (ginfo->players[i].UCID == 0)
            break;

    // If i reached MAX_PLAYERS then we have more players than what the application expects.
    // MAX_PLAYERS should be set to the actual maximum number of drivers that the server allows.
    if (i == MAX_PLAYERS)
    {
        //cout << "NO MORE PEOPLE ALLOWED!!!" << endl;
        return;
    }

    // Copy all the player data we need into the ginfo->players[] array
    //cout << "(Initializing player data into global struct)" << endl;
    strcpy(ginfo->players[i].UName, pack_ncn->UName);
    strcpy(ginfo->players[i].PName, pack_ncn->PName);
    ginfo->players[i].UCID = pack_ncn->UCID;
    //cout << "pack_ncn->UCID: " << (int)pack_ncn->UCID << endl;
    ginfo->players[i].Admin = pack_ncn->Admin;

    for (int j=0; j<64; j++)
    {
        if (strcmp(ginfo->disconnections[j], ginfo->players[i].UName) == 0)
        {
            if (ginfo->hardcore)
                ginfo->players[i].invalidpit = 1;
            else
                memset(ginfo->disconnections[j], 0, sizeof(ginfo->disconnections[j]));

            break;
        }
    }

    for (int j=0; j<50; j++)
    {
        if (strcmp(ginfo->players[i].UName, ginfo->directors[j]) == 0)
        {
            ginfo->players[i].Director = 1;
            ////cout << "\n JUGADOR: " << ginfo->players[i].UName << " confirmado como DIRECTOR: 1" << endl;
        }
    }

    //cout << "NEW player connected!" << endl;
    //cout << "UName:" << ginfo->players[i].UName << endl;
    //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
    //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;

    // Send a welcome message
    struct IS_MTC pack_mtc;
    memset(&pack_mtc, 0, sizeof(struct IS_MTC));
    pack_mtc.Size = sizeof(struct IS_MTC);
    pack_mtc.Type = ISP_MTC;
    pack_mtc.UCID = ginfo->players[i].UCID;
    strcpy(pack_mtc.Text, "^3Welcome! ^6Event Control v0.43 ^3is active");
    insim.send_packet(&pack_mtc);

    // Send the help screen as the player connects
    help_screen(&ginfo->players[i]);
}


/**
* New player joining race or leaving pits
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_npl (struct global_info *ginfo)
{
    //cout << " * IS_NPL (joining race or leaving pits)" << endl;
    int i;

    struct IS_NPL *pack_npl = (struct IS_NPL*)insim.get_packet();

    // Find player using UCID and update his PLID
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID == pack_npl->UCID)
        {
            //cout << "ginfo->players[" << i << "].UCID found" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            ginfo->players[i].PLID = pack_npl->PLID;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << " (new one if joining, or the same he had if exiting pits)" << endl;
            break;
        }
    }

    // We request this packets at the beginning of the main, to check all the players who were in the server when the application
    // + was started. We send this players to spectators so they start clean.
    if (pack_npl->ReqI == 1)
    {
        struct IS_MST pack_mst;
        memset(&pack_mst, 0, sizeof(struct IS_MST));
        pack_mst.Size = sizeof(struct IS_MST);
        pack_mst.Type = ISP_MST;
        strcpy(pack_mst.Msg, "/spec ");
        strcpy(pack_mst.Msg + 6, pack_npl->PName);
        insim.send_packet(&pack_mst);

        //cout << "(Player " << pack_npl->PName << " was in the grid, taken to spectators)" << endl;
    }

    // If player did an invalid pit, he's taken to spectators
    if (ginfo->players[i].invalidpit == 1)
    {
        char textorejoin[100];
        memset(textorejoin, 0, 100);

        struct IS_MST pack_mst;
        memset(&pack_mst, 0, sizeof(struct IS_MST));
        pack_mst.Size = sizeof(struct IS_MST);
        pack_mst.Type = ISP_MST;
        strcpy(pack_mst.Msg, "/spec ");
        strcpy(pack_mst.Msg + 6, pack_npl->PName);
        insim.send_packet(&pack_mst);

        struct IS_BTN pack_btn;
        memset(&pack_btn, 0, sizeof(struct IS_BTN));
        pack_btn.Size = sizeof(struct IS_BTN);
        pack_btn.Type = ISP_BTN;
        pack_btn.ReqI = ginfo->players[i].PLID;
        pack_btn.UCID = ginfo->players[i].UCID;

        pack_btn.ClickID = 235;
        pack_btn.BStyle = ISB_LIGHT;                // Light frame for main text
        pack_btn.L = 54;
        pack_btn.T = 80;
        pack_btn.W = 92;
        pack_btn.H = 40;
        insim.send_packet(&pack_btn);

        if (ginfo->trackclosed)
        {
            pack_btn.ClickID = 234;
            pack_btn.BStyle = 0;
            pack_btn.L = 73;
            pack_btn.T = 80;
            pack_btn.W = 54;
            pack_btn.H = 14;
            strcpy(textorejoin, "^1TRACK CLOSED!");
            memcpy(pack_btn.Text, textorejoin, sizeof(pack_btn.Text)-1);
            insim.send_packet(&pack_btn);

            pack_btn.ClickID = 233;
            pack_btn.L = 55;
            pack_btn.T = 94;
            pack_btn.W = 90;
            pack_btn.H = 12;
            strcpy(textorejoin, "^3You can't rejoin until the track is open");
            memcpy(pack_btn.Text, textorejoin, sizeof(pack_btn.Text)-1);
            insim.send_packet(&pack_btn);
        }
        else
        {
            pack_btn.ClickID = 234;
            pack_btn.BStyle = 0;
            pack_btn.L = 73;
            pack_btn.T = 80;
            pack_btn.W = 54;
            pack_btn.H = 14;
            strcpy(textorejoin, "^1INVALID REJOIN!");
            memcpy(pack_btn.Text, textorejoin, sizeof(pack_btn.Text)-1);
            insim.send_packet(&pack_btn);

            pack_btn.ClickID = 233;
            pack_btn.L = 55;
            pack_btn.T = 94;
            pack_btn.W = 90;
            pack_btn.H = 12;
            strcpy(textorejoin, "^3You can't rejoin until qualify ends");
            memcpy(pack_btn.Text, textorejoin, sizeof(pack_btn.Text)-1);
            insim.send_packet(&pack_btn);

        }

        pack_btn.ClickID = 232;                      // "Close" button
        pack_btn.BStyle = ISB_DARK + ISB_CLICK;      // dark + clickable
        pack_btn.L = 92;
        pack_btn.T = 108;
        pack_btn.W = 16;
        pack_btn.H = 10;
        strcpy(pack_btn.Text, "close");
        insim.send_packet(&pack_btn);

        //cout << "(Player " << pack_npl->PName << " invalid rejoin, taken to spectators)" << endl;
    }
}


/**
* Actions performed when pit lane is left or entered
* @param    ginfo   Struct that contains all the info stored by the application
*/

void case_pla (struct global_info *ginfo)
{
    int i;

    struct IS_PLA *pack_pla = (struct IS_PLA*)insim.get_packet();

    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].PLID == pack_pla->PLID)
        {
            //cout << "ginfo->players[" << i << "].PLID encontrado" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            //cout << "Fact: " << (int)pack_pla->Fact << endl;
            break;
        }
    }

    if ((pack_pla->Fact == PITLANE_EXIT) && (ginfo->hardcore == 1))
    {
        ginfo->players[i].invalidpit = 1;
        //cout << "invalidpit: 1" << endl;

        char textoflying[100];
        memset(textoflying, 0, 100);

        ginfo->players[i].flyinglap = 1;

        struct IS_BTN pack_btn;
        memset(&pack_btn, 0, sizeof(struct IS_BTN));
        pack_btn.Size = sizeof(struct IS_BTN);
        pack_btn.Type = ISP_BTN;
        pack_btn.ReqI = ginfo->players[i].PLID;
        pack_btn.UCID = ginfo->players[i].UCID;
        pack_btn.ClickID = 237;
        pack_btn.L = 75;
        pack_btn.T = 31;
        pack_btn.W = 50;
        pack_btn.H = 9;
        strcpy(textoflying, "^4OUT LAP - GIVE WAY");
        memcpy(pack_btn.Text, textoflying, sizeof(pack_btn.Text)-1);
        insim.send_packet(&pack_btn);

        pack_btn.ClickID = 236;
        pack_btn.T = 27;
        pack_btn.L = 65;
        pack_btn.W = 70;
        pack_btn.H = 5;
        strcpy(textoflying, "^7Spectating or shift+P NOT allowed now");
        memcpy(pack_btn.Text, textoflying, sizeof(pack_btn.Text)-1);
        insim.send_packet(&pack_btn);
    }
    if ((pack_pla->Fact == PITLANE_ENTER) && (ginfo->hardcore) && (ginfo->trackclosed == 0) && (ginfo->players[i].hardmsg == 0))
    {
        ginfo->players[i].hardmsg = 1;

        char textoflying[100];
        memset(textoflying, 0, 100);

        struct IS_BTN pack_btn;
        memset(&pack_btn, 0, sizeof(struct IS_BTN));
        pack_btn.Size = sizeof(struct IS_BTN);
        pack_btn.Type = ISP_BTN;
        pack_btn.ReqI = ginfo->players[i].PLID;
        pack_btn.UCID = ginfo->players[i].UCID;
        pack_btn.ClickID = 238;
        pack_btn.BStyle = ISB_DARK;
        pack_btn.L = 65;
        pack_btn.T = 65;
        pack_btn.W = 70;
        pack_btn.H = 13;
        strcpy(textoflying, "^1HARDCORE QUALIFY SESSION!");
        memcpy(pack_btn.Text, textoflying, sizeof(pack_btn.Text)-1);
        insim.send_packet(&pack_btn);

        // Create a thread to clear this button after 5 seconds. Will call clear_msg()
        int rc;
        pthread_t thread;
        rc = pthread_create(&thread, NULL, clear_msg, (void *)&ginfo->players[i].UCID);
        if (rc){
            printf("ERROR; return code from pthread_create() is %d\n", rc);
            exit(-1);
        }
    }
}


/**
* Player leaves race (spectates - loses slot)
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_pll (struct global_info *ginfo)
{
    //cout << " * IS_PLL (player leaves race)" << endl;
    int i;

    struct IS_PLL *pack_pll = (struct IS_PLL*)insim.get_packet();

    // Find player and set his PLID to 0
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].PLID == pack_pll->PLID)
        {
            //cout << "ginfo->players[" << i << "].PLID found" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            ginfo->players[i].PLID = 0;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << " (set to zero)" << endl;
            break;
        }
    }

    // If player was on his out lap, clear the warning message
    if (ginfo->players[i].flyinglap == 1)
    {
        ginfo->players[i].flyinglap = 0;
        erase_flyinglap(&ginfo->players[i]);
    }

    // If track was on hardcore session, clear no pit message
    if (ginfo->hardcore == 1)
        erase_nopit(&ginfo->players[i]);
}

/**
* Player enter pits (keeps slot)
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_plp (struct global_info *ginfo)
{
    //cout << " * IS_PLP (player enter pits)" << endl;
    int i;

    struct IS_PLP *pack_plp = (struct IS_PLP*)insim.get_packet();

    // Find player
    for (i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].PLID == pack_plp->PLID)
        {
            //cout << "ginfo->players[" << i << "].PLID found" << endl;
            //cout << "UName: " << ginfo->players[i].UName << endl;
            //cout << "UCID: " << (int)ginfo->players[i].UCID << endl;
            //cout << "PLID: " << (int)ginfo->players[i].PLID << endl;
            break;
        }
    }

    // If player was on his out lap, clear the warning message
    if (ginfo->players[i].flyinglap == 1)
    {
        ginfo->players[i].flyinglap = 0;
        erase_flyinglap(&ginfo->players[i]);
    }

    // If track was on hardcore session, clear no pit message
    if (ginfo->hardcore == 1)
        erase_nopit(&ginfo->players[i]);
}


/**
* When a IS_REO packet arrives means that reorder is going on
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_reo (struct global_info *ginfo)
{
    //cout << " * IS_REO (reorder)" << endl;

    struct IS_REO *pack_reo = (struct IS_REO*)insim.get_packet();

    if ((pack_reo->ReqI != 0) && (pack_reo->ReqI < 32))
    {
        byte aux = 0;
        byte numc = pack_reo->ReqI;
        char numctext[3];

        if (pack_reo->NumP < numc)
            numc = pack_reo->NumP;

        // Send /msg message to announce that grid is going to be reversed
        struct IS_MST pack_mst;
        memset(&pack_mst, 0, sizeof(struct IS_MST));
        pack_mst.Size = sizeof(struct IS_MST);
        pack_mst.Type = ISP_MST;
        strcpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** REVERSING TOP ");
        sprintf(numctext,"%d",numc);
        strcat(pack_mst.Msg, numctext);
        strcat(pack_mst.Msg, " DRIVERS! ***");
        insim.send_packet(&pack_mst);

        //cout << "\n EL PAQUETE FUE SOLICITADO!!!" << endl;

        //cout << "\n BEFORE REORDER:" << endl;
        for (int i=0; i<32; i++)
        {
            //cout << "PLID[" << i << "]: " << (int)pack_reo->PLID[i] << endl;
        }

        for (int i=0; i<numc/2; i++)
        {
            aux = pack_reo->PLID[i];
            pack_reo->PLID[i] = pack_reo->PLID[numc-i-1];
            pack_reo->PLID[numc-i-1] = aux;
        }

        //cout << "\n AFTER REORDER:" << endl;
        for (int i=0; i<32; i++)
        {
            //cout << "PLID[" << i << "]: " << (int)pack_reo->PLID[i] << endl;
        }

        // Send IS_REO packet with new order
        pack_reo->Size = sizeof(struct IS_REO);
        insim.send_packet(pack_reo);
    }
}


/**
* When a IS_RST packet arrives means that practice/qualifying/race is restarting.
* We must reset everybody's invalidpit status to zero.
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_rst (struct global_info *ginfo)
{
    //cout << " * IS_RST (session restart)" << endl;

    struct IS_RST *pack_rst = (struct IS_RST*)insim.get_packet();

    // Reset everybody's invalidpit status to zero
    //cout << "Reseting everybody's invalidpit status to 0 (zero)" << endl;
    for (int i=0; i < MAX_PLAYERS; i++)
    {
        if (ginfo->players[i].UCID != 0)
        {
            erase_flyinglap(&ginfo->players[i]);
            erase_nopit(&ginfo->players[i]);
        }
        ginfo->players[i].invalidpit = 0;
        ginfo->players[i].hardmsg = 0;
    }

    if (pack_rst->QualMins == 0)
    {
        ginfo->trackclosed = 0;
        ginfo->hardcore = 0;
        //cout << "ginfo->trackclosed: 0" << endl;
        //cout << "ginfo->hardcore: 0" << endl;
    }
    else
    {
        // Send /msg message to announce the kind of session
        struct IS_MST pack_mst;
        memset(&pack_mst, 0, sizeof(struct IS_MST));
        pack_mst.Size = sizeof(struct IS_MST);
        pack_mst.Type = ISP_MST;

        if (ginfo->hardcore)
        {
            strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** ^1HARDCORE ^3SESSION STARTS NOW! ***", 63);
            insim.send_packet(&pack_mst);
        }
        else
        {
            strncpy(pack_mst.Msg, "/msg ^6Event Control: ^3*** ^2SOFT ^3SESSION STARTS NOW! ***", 63);
            insim.send_packet(&pack_mst);
        }
    }
}


/**
* When a IS_STA packet arrives means that session status has changed
* If we are on lobby screen, reset everything
* @param    ginfo   Struct that contains all the info stored by the application
*/
void case_sta (struct global_info *ginfo)
{
    //cout << " * IS_STA (session status)" << endl;

    struct IS_STA *pack_sta = (struct IS_STA*)insim.get_packet();

    // If no race in progress, then we're on the lobby screen
    if (pack_sta->RaceInProg == 0)
    {
        // Reset everybody's invalidpit status to zero
        //cout << "Reseting everybody's invalidpit status to 0 (zero)" << endl;
        for (int i=0; i < MAX_PLAYERS; i++)
        {
            erase_flyinglap(&ginfo->players[i]);
            ginfo->players[i].invalidpit = 0;
        }

        ginfo->trackclosed = 0;
        //cout << "ginfo->trackclosed: 0" << endl;
    }
}


/**
* Main program. Receives arguments used in the connection to the server.
* @param    argc    Number of arguments in the application call
* @param    argv    Arguments received in the application call.
*/
int main (int argc, char **argv)
{
    // Error check var used when calling next_packet()
    int error_ch;

    // Create the main global_info struct and clear it completely to zeroes.
    struct global_info ginfo;
    memset(&ginfo, 0, sizeof(struct global_info));

    // Read the config file
    if (!(read_config(&ginfo)))
    {
        cerr << "\n * Not enough arguments! *" << endl;
        cerr << "\n * Check config.cfg for possible errors * " << endl;
        cerr << "\n PRESS RETURN KEY TO FINISH" << endl;
        getchar();
        return -1;
    }

    // We create an IS_VER packet to receive the response from insim.init()
    struct IS_VER pack_ver;

    // Initialize the connection to InSim:
    if (insim.init (ginfo.config_data[0], (word)atoi(ginfo.config_data[1]), IS_PRODUCT_NAME, ginfo.config_data[2], &pack_ver, '!') < 0)
    {
        cerr << "\n * Error during initialization * " << endl;
        cerr << "\n * Check config.cfg for possible errors * " << endl;
        cerr << "\n PRESS RETURN KEY TO FINISH" << endl;
        getchar();
        return -1;
    }

    cout << "\n********************************\n" << endl;
    cout << "LFS Version: " << pack_ver.Version << endl;
    cout << "InSim Version: " << pack_ver.InSimVer << endl;
    cout << "\n********************************\n" << endl;

    // Initialize Event Control
    eventcontrol_init(&ginfo);

    // Main loop. When ok is 0 we'll exit this loop
    int ok = 1;
    while(ok > 0)
    {
        // Get next packet ready
        if (error_ch = insim.next_packet() < 0)
        {
            cerr << "\n * Error getting next packet * " << endl;
            cerr << "\n PRESS RETURN KEY TO FINISH" << endl;
            getchar();
            exit_message(&ginfo,1);
            return error_ch;
        }

        // In case_mso the user can request to close the application or turn ON or OFF
        if (insim.peek_packet() == ISP_MSO)
        {
            ok = case_mso (&ginfo);
        }
        // The rest of the cases
        else
        {
            if (!(ginfo.off))
            {
                // Peek the next packet to know its type, and call the matching function
                //cout << " \n***********************************************************" << endl;
                //cout << " * Received packet of type: " << (int)insim.peek_packet() << endl;
                switch (insim.peek_packet())
                {
                    case ISP_BTC:
                        case_btc (&ginfo);
                        break;
                    case ISP_CNL:
                        case_cnl (&ginfo);
                        break;
                    case ISP_CPR:
                        case_cpr (&ginfo);
                        break;
                    case ISP_LAP:
                        case_lap (&ginfo);
                        break;
                    case ISP_NCN:
                        case_ncn (&ginfo);
                        break;
                    case ISP_NPL:
                        case_npl (&ginfo);
                        break;
                    case ISP_PLA:
                        case_pla (&ginfo);
                        break;
                    case ISP_PLL:
                        case_pll (&ginfo);
                        break;
                    case ISP_PLP:
                        case_plp (&ginfo);
                        break;
                    case ISP_REO:
                        case_reo (&ginfo);
                        break;
                    case ISP_RST:
                        case_rst (&ginfo);
                        break;
                    case ISP_STA:
                        case_sta (&ginfo);
                        break;
                }
            }
        }
    }

    // The main loop ended: close connection to insim
    if (insim.isclose() < 0)
    {
        cerr << "\n * Error closing connection * " << endl;
        cerr << "\n PRESS RETURN KEY TO FINISH" << endl;
        getchar();
        return -1;
    }

    return 0;
}
